package app.firezone.synk.network.Adapter;

import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;
import androidx.recyclerview.widget.RecyclerView;

import java.util.List;

import app.firezone.synk.network.R;
import app.firezone.synk.network.model.SlotItem;

public class SlotAdapter extends RecyclerView.Adapter<SlotAdapter.SlotViewHolder> {

    private final List<SlotItem> slotList;
    private final String teamType;
    private final Context context;
    private int maxSelection;

    public SlotAdapter(Context context, List<SlotItem> slotList, String teamType) {
        this.context = context;
        this.slotList = slotList;
        this.teamType = teamType;

        if (teamType.equalsIgnoreCase("Solo")) {
            maxSelection = 1;
        } else if (teamType.equalsIgnoreCase("Duo")) {
            maxSelection = 2;
        } else {
            maxSelection = 4;
        }
    }

    @NonNull
    @Override
    public SlotViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View view = LayoutInflater.from(context).inflate(R.layout.item_slot, parent, false);
        return new SlotViewHolder(view);
    }

    @Override
    public void onBindViewHolder(@NonNull SlotViewHolder holder, int position) {
        SlotItem slot = slotList.get(position);
        holder.tvSlotNumber.setText("" + slot.getSlotNumber());

        setUpCheckbox(holder.cbA, slot, 'A');
        setUpCheckbox(holder.cbB, slot, 'B');
        setUpCheckbox(holder.cbC, slot, 'C');
        setUpCheckbox(holder.cbD, slot, 'D');

        holder.cbA.setVisibility(View.VISIBLE);

        if (teamType.equalsIgnoreCase("Duo")) {
            holder.cbB.setVisibility(View.VISIBLE);
            holder.cbC.setVisibility(View.GONE);
            holder.cbD.setVisibility(View.GONE);
        } else if (teamType.equalsIgnoreCase("Squad")) {
            holder.cbB.setVisibility(View.VISIBLE);
            holder.cbC.setVisibility(View.VISIBLE);
            holder.cbD.setVisibility(View.VISIBLE);
        } else {
            holder.cbB.setVisibility(View.GONE);
            holder.cbC.setVisibility(View.GONE);
            holder.cbD.setVisibility(View.GONE);
        }
    }

    private void setUpCheckbox(CheckBox checkBox, SlotItem slot, char label) {
        int red = ContextCompat.getColor(context, android.R.color.holo_red_dark);
        int joinedTint = ContextCompat.getColor(context, R.color.joinedTint);

        checkBox.setOnCheckedChangeListener(null); // remove any previous listener

        boolean isChecked = false;
        boolean isJoined = false;

        switch (label) {
            case 'A':
                isChecked = slot.isCheckedA;
                isJoined = slot.isJoinedA;
                break;
            case 'B':
                isChecked = slot.isCheckedB;
                isJoined = slot.isJoinedB;
                break;
            case 'C':
                isChecked = slot.isCheckedC;
                isJoined = slot.isJoinedC;
                break;
            case 'D':
                isChecked = slot.isCheckedD;
                isJoined = slot.isJoinedD;
                break;
        }

        checkBox.setText(String.valueOf(label));

        if (isJoined) {
            checkBox.setChecked(true);
            checkBox.setEnabled(false);
            checkBox.setButtonTintList(ColorStateList.valueOf(joinedTint));
            checkBox.setTextColor(Color.GRAY);
        } else {
            checkBox.setChecked(isChecked);
            checkBox.setEnabled(true);
            checkBox.setButtonTintList(ColorStateList.valueOf(red));
            checkBox.setTextColor(Color.BLACK);

            checkBox.setOnCheckedChangeListener((buttonView, isCheckedNow) -> {
                int totalChecked = getTotalChecked();

                if (isCheckedNow && totalChecked >= maxSelection) {
                    buttonView.setChecked(false);
                    Toast.makeText(context, "You can only select " + maxSelection + " slot(s)", Toast.LENGTH_SHORT).show();
                    return;
                }

                switch (label) {
                    case 'A': slot.isCheckedA = isCheckedNow; break;
                    case 'B': slot.isCheckedB = isCheckedNow; break;
                    case 'C': slot.isCheckedC = isCheckedNow; break;
                    case 'D': slot.isCheckedD = isCheckedNow; break;
                }
            });
        }
    }

    private int getTotalChecked() {
        int count = 0;
        for (SlotItem item : slotList) {
            if (item.isCheckedA && !item.isJoinedA) count++;
            if (item.isCheckedB && !item.isJoinedB) count++;
            if (item.isCheckedC && !item.isJoinedC) count++;
            if (item.isCheckedD && !item.isJoinedD) count++;
        }
        return count;
    }

    @Override
    public int getItemCount() {
        return slotList.size();
    }

    static class SlotViewHolder extends RecyclerView.ViewHolder {
        TextView tvSlotNumber;
        CheckBox cbA, cbB, cbC, cbD;

        public SlotViewHolder(@NonNull View itemView) {
            super(itemView);
            tvSlotNumber = itemView.findViewById(R.id.tvSlotNumber);
            cbA = itemView.findViewById(R.id.cbA);
            cbB = itemView.findViewById(R.id.cbB);
            cbC = itemView.findViewById(R.id.cbC);
            cbD = itemView.findViewById(R.id.cbD);
        }
    }
}
